<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // disable direct access
}

use function FortAwesome\fa;

if ( ! class_exists('Mega_Menu_Font_Awesome_5') ) :

/**
 *
 */
class Mega_Menu_Font_Awesome_5 {

	/**
	 * Constructor
	 *
	 * @since 1.8
	 */
	public function __construct() {
        add_filter( 'megamenu_scss_variables', array( $this, 'add_fa5_scss_vars'), 10, 4 );
		add_filter( 'megamenu_load_scss_file_contents', array( $this, 'append_fontawesome_scss'), 10 );
		add_filter( 'megamenu_icon_tabs', array( $this, 'font_awesome_selector'), 99, 5 );
		add_action( 'megamenu_enqueue_public_scripts', array ( $this, 'enqueue_public_scripts'), 10 );
        add_action( "megamenu_admin_scripts", array( $this, 'enqueue_admin_styles') );
		add_action( "admin_print_scripts-nav-menus.php", array( $this, 'enqueue_admin_styles' ) );
	}

	
	/**
	 * Returns true when FontAwesome 5 Pro is installed/enabled
	 */
	public function use_pro() {
		if ( defined( "MEGAMENU_PRO_USE_FONTAWESOME5_PRO" ) ) {
			return MEGAMENU_PRO_USE_FONTAWESOME5_PRO;
		}

		if ( function_exists( 'FortAwesome\fa' ) && fa()->pro() ) {
			return true;
		}

		return false;
	}


    /**
     * Enqueue required CSS and JS for Mega Menu
     *
     */
    public function enqueue_admin_styles( $hook ) {
        wp_enqueue_style( 'megamenu-fontawesome5', plugins_url( 'css/all.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
    }


    /**
     * Make the $font_awesome_family scss variable available
     *
     * @param array $vars
     * @param string $location
     * @param array $theme
     * @param int $menu_id
     * @return array
     */
    public function add_fa5_scss_vars( $vars, $location, $theme, $menu_id ) {
    	if ( $this->use_pro() ) {
        	$vars['font_awesome_family'] = "'Font Awesome 5 Pro'";
    	} else {
        	$vars['font_awesome_family'] = "'Font Awesome 5 Free'";
    	}

        return $vars;
    }


	/**
	 * Add the CSS required to display fontawesome icons to the main SCSS file
	 *
	 * @since 1.8
	 * @param string $scss
	 * @return string
	 */
	public function append_fontawesome_scss( $scss ) {
		$path = trailingslashit( plugin_dir_path( __FILE__ ) ) . 'scss/fontawesome.scss';
		$contents = file_get_contents( $path );
 		return $scss . $contents;
	}


	/**
	 * Enqueue fontawesome CSS
	 *
	 * @since 1.8
	 */
	public function enqueue_public_scripts() {
		$settings = get_option("megamenu_settings");

        if ( is_array( $settings ) && isset( $settings['enqueue_fa_5'] ) && $settings['enqueue_fa_5'] == 'disabled' ) {
        	return;
        }

        wp_enqueue_style( 'megamenu-fontawesome5', plugins_url( 'css/all.min.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
	}


	/**
	 * Generate HTML for the font awesome selector
	 *
	 * @since 1.8
	 * @param array $tabs
	 * @param int $menu_item_id
	 * @param int $menu_id
	 * @param int $menu_item_depth
	 * @param array $menu_item_meta
	 * @return array
	 */
	public function font_awesome_selector( $tabs, $menu_item_id, $menu_id, $menu_item_depth, $menu_item_meta ) {
		$is_pro = $this->use_pro() ? "pro" : "";
		$settings = get_option("megamenu_settings");
        $css_version = get_option("megamenu_pro_css_version");
        $html = "";

        if ( is_array( $settings ) && isset( $settings['enqueue_fa_5'] ) && $settings['enqueue_fa_5'] == 'disabled' ) {
        	$html .= "<div class='notice notice-warning'>" . __("Font Awesome 5 has been dequeued under Mega Menu > General Settings. You will need to ensure that Font Awesome 5 is enqueued on your site using an alternative method.", "megamenupro") . "</div>";
        }

        foreach ( $this->icons() as $code => $class ) {

            $bits = explode( "-", $code );
            $code = "&#x" . $bits[1] . "";
            $type = $bits[0];

            $style_bits = explode( " ", $class);
            $style = $style_bits[0];

            $html .= "<div class='{$style} {$is_pro}'>";
            $html .= "    <input class='radio' id='{$class}' type='radio' rel='{$code}' name='settings[icon]' value='{$class}' " . checked( $menu_item_meta['icon'], $class, false ) . " />";
            $html .= "    <label rel='{$code}' for='{$class}' title='{$class}'></label>";
            $html .= "</div>";

        }

        $icon_prefix = "";

        if ( isset( $menu_item_meta['icon'] ) ) {
        	$icon_prefix = substr( $menu_item_meta['icon'], 0, 3 );
        }

        $title = __("Font Awesome 5", "megamenupro");

        if ( $this->use_pro() ) {
        	$title .= " Pro";
        }

		$insert['fontawesome5'] = array(
			'title' => $title,
			'active' => isset( $menu_item_meta['icon'] ) && in_array( $icon_prefix, array( 'fab', 'fas', 'far', 'fal' ) ),
			'content' => $html
		);

		array_splice( $tabs, 2, 0, $insert );
		
		return $tabs;

	}

	/**
	 * Return an array of Font Awesome icons
	 *
	 * @since 1.8
	 * @return array $icons
	 */
	public function icons() {

		$solid_icons = array(
			"fas-f641" => "fas fa-ad",
			"fas-f2b9" => "fas fa-address-book",
			"fas-f2bb" => "fas fa-address-card",
			"fas-f042" => "fas fa-adjust",
			"fas-f5d0" => "fas fa-air-freshener",
			"fas-f037" => "fas fa-align-center",
			"fas-f039" => "fas fa-align-justify",
			"fas-f036" => "fas fa-align-left",
			"fas-f038" => "fas fa-align-right",
			"fas-f461" => "fas fa-allergies",
			"fas-f0f9" => "fas fa-ambulance",
			"fas-f2a3" => "fas fa-american-sign-language-interpreting",
			"fas-f13d" => "fas fa-anchor",
			"fas-f103" => "fas fa-angle-double-down",
			"fas-f100" => "fas fa-angle-double-left",
			"fas-f101" => "fas fa-angle-double-right",
			"fas-f102" => "fas fa-angle-double-up",
			"fas-f107" => "fas fa-angle-down",
			"fas-f104" => "fas fa-angle-left",
			"fas-f105" => "fas fa-angle-right",
			"fas-f106" => "fas fa-angle-up",
			"fas-f556" => "fas fa-angry",
			"fas-f644" => "fas fa-ankh",
			"fas-f5d1" => "fas fa-apple-alt",
			"fas-f187" => "fas fa-archive",
			"fas-f557" => "fas fa-archway",
			"fas-f358" => "fas fa-arrow-alt-circle-down",
			"fas-f359" => "fas fa-arrow-alt-circle-left",
			"fas-f35a" => "fas fa-arrow-alt-circle-right",
			"fas-f35b" => "fas fa-arrow-alt-circle-up",
			"fas-f0ab" => "fas fa-arrow-circle-down",
			"fas-f0a8" => "fas fa-arrow-circle-left",
			"fas-f0a9" => "fas fa-arrow-circle-right",
			"fas-f0aa" => "fas fa-arrow-circle-up",
			"fas-f063" => "fas fa-arrow-down",
			"fas-f060" => "fas fa-arrow-left",
			"fas-f061" => "fas fa-arrow-right",
			"fas-f062" => "fas fa-arrow-up",
			"fas-f0b2" => "fas fa-arrows-alt",
			"fas-f337" => "fas fa-arrows-alt-h",
			"fas-f338" => "fas fa-arrows-alt-v",
			"fas-f2a2" => "fas fa-assistive-listening-systems",
			"fas-f069" => "fas fa-asterisk",
			"fas-f1fa" => "fas fa-at",
			"fas-f558" => "fas fa-atlas",
			"fas-f5d2" => "fas fa-atom",
			"fas-f29e" => "fas fa-audio-description",
			"fas-f559" => "fas fa-award",
			"fas-f77c" => "fas fa-baby",
			"fas-f77d" => "fas fa-baby-carriage",
			"fas-f55a" => "fas fa-backspace",
			"fas-f04a" => "fas fa-backward",
			"fas-f7e5" => "fas fa-bacon",
			"fas-f666" => "fas fa-bahai",
			"fas-f24e" => "fas fa-balance-scale",
			"fas-f515" => "fas fa-balance-scale-left",
			"fas-f516" => "fas fa-balance-scale-right",
			"fas-f05e" => "fas fa-ban",
			"fas-f462" => "fas fa-band-aid",
			"fas-f02a" => "fas fa-barcode",
			"fas-f0c9" => "fas fa-bars",
			"fas-f433" => "fas fa-baseball-ball",
			"fas-f434" => "fas fa-basketball-ball",
			"fas-f2cd" => "fas fa-bath",
			"fas-f244" => "fas fa-battery-empty",
			"fas-f240" => "fas fa-battery-full",
			"fas-f242" => "fas fa-battery-half",
			"fas-f243" => "fas fa-battery-quarter",
			"fas-f241" => "fas fa-battery-three-quarters",
			"fas-f236" => "fas fa-bed",
			"fas-f0fc" => "fas fa-beer",
			"fas-f0f3" => "fas fa-bell",
			"fas-f1f6" => "fas fa-bell-slash",
			"fas-f55b" => "fas fa-bezier-curve",
			"fas-f647" => "fas fa-bible",
			"fas-f206" => "fas fa-bicycle",
			"fas-f84a" => "fas fa-biking",
			"fas-f1e5" => "fas fa-binoculars",
			"fas-f780" => "fas fa-biohazard",
			"fas-f1fd" => "fas fa-birthday-cake",
			"fas-f517" => "fas fa-blender",
			"fas-f6b6" => "fas fa-blender-phone",
			"fas-f29d" => "fas fa-blind",
			"fas-f781" => "fas fa-blog",
			"fas-f032" => "fas fa-bold",
			"fas-f0e7" => "fas fa-bolt",
			"fas-f1e2" => "fas fa-bomb",
			"fas-f5d7" => "fas fa-bone",
			"fas-f55c" => "fas fa-bong",
			"fas-f02d" => "fas fa-book",
			"fas-f6b7" => "fas fa-book-dead",
			"fas-f7e6" => "fas fa-book-medical",
			"fas-f518" => "fas fa-book-open",
			"fas-f5da" => "fas fa-book-reader",
			"fas-f02e" => "fas fa-bookmark",
			"fas-f84c" => "fas fa-border-all",
			"fas-f850" => "fas fa-border-none",
			"fas-f853" => "fas fa-border-style",
			"fas-f436" => "fas fa-bowling-ball",
			"fas-f466" => "fas fa-box",
			"fas-f49e" => "fas fa-box-open",
			"fas-f95b" => "fas fa-box-tissue",
			"fas-f468" => "fas fa-boxes",
			"fas-f2a1" => "fas fa-braille",
			"fas-f5dc" => "fas fa-brain",
			"fas-f7ec" => "fas fa-bread-slice",
			"fas-f0b1" => "fas fa-briefcase",
			"fas-f469" => "fas fa-briefcase-medical",
			"fas-f519" => "fas fa-broadcast-tower",
			"fas-f51a" => "fas fa-broom",
			"fas-f55d" => "fas fa-brush",
			"fas-f188" => "fas fa-bug",
			"fas-f1ad" => "fas fa-building",
			"fas-f0a1" => "fas fa-bullhorn",
			"fas-f140" => "fas fa-bullseye",
			"fas-f46a" => "fas fa-burn",
			"fas-f207" => "fas fa-bus",
			"fas-f55e" => "fas fa-bus-alt",
			"fas-f64a" => "fas fa-business-time",
			"fas-f1ec" => "fas fa-calculator",
			"fas-f133" => "fas fa-calendar",
			"fas-f073" => "fas fa-calendar-alt",
			"fas-f274" => "fas fa-calendar-check",
			"fas-f783" => "fas fa-calendar-day",
			"fas-f272" => "fas fa-calendar-minus",
			"fas-f271" => "fas fa-calendar-plus",
			"fas-f273" => "fas fa-calendar-times",
			"fas-f784" => "fas fa-calendar-week",
			"fas-f030" => "fas fa-camera",
			"fas-f083" => "fas fa-camera-retro",
			"fas-f6bb" => "fas fa-campground",
			"fas-f786" => "fas fa-candy-cane",
			"fas-f55f" => "fas fa-cannabis",
			"fas-f46b" => "fas fa-capsules",
			"fas-f1b9" => "fas fa-car",
			"fas-f5de" => "fas fa-car-alt",
			"fas-f5df" => "fas fa-car-battery",
			"fas-f5e1" => "fas fa-car-crash",
			"fas-f5e4" => "fas fa-car-side",
			"fas-f8ff" => "fas fa-caravan",
			"fas-f0d7" => "fas fa-caret-down",
			"fas-f0d9" => "fas fa-caret-left",
			"fas-f0da" => "fas fa-caret-right",
			"fas-f150" => "fas fa-caret-square-down",
			"fas-f191" => "fas fa-caret-square-left",
			"fas-f152" => "fas fa-caret-square-right",
			"fas-f151" => "fas fa-caret-square-up",
			"fas-f0d8" => "fas fa-caret-up",
			"fas-f787" => "fas fa-carrot",
			"fas-f218" => "fas fa-cart-arrow-down",
			"fas-f217" => "fas fa-cart-plus",
			"fas-f788" => "fas fa-cash-register",
			"fas-f6be" => "fas fa-cat",
			"fas-f0a3" => "fas fa-certificate",
			"fas-f6c0" => "fas fa-chair",
			"fas-f51b" => "fas fa-chalkboard",
			"fas-f51c" => "fas fa-chalkboard-teacher",
			"fas-f5e7" => "fas fa-charging-station",
			"fas-f1fe" => "fas fa-chart-area",
			"fas-f080" => "fas fa-chart-bar",
			"fas-f201" => "fas fa-chart-line",
			"fas-f200" => "fas fa-chart-pie",
			"fas-f00c" => "fas fa-check",
			"fas-f058" => "fas fa-check-circle",
			"fas-f560" => "fas fa-check-double",
			"fas-f14a" => "fas fa-check-square",
			"fas-f7ef" => "fas fa-cheese",
			"fas-f439" => "fas fa-chess",
			"fas-f43a" => "fas fa-chess-bishop",
			"fas-f43c" => "fas fa-chess-board",
			"fas-f43f" => "fas fa-chess-king",
			"fas-f441" => "fas fa-chess-knight",
			"fas-f443" => "fas fa-chess-pawn",
			"fas-f445" => "fas fa-chess-queen",
			"fas-f447" => "fas fa-chess-rook",
			"fas-f13a" => "fas fa-chevron-circle-down",
			"fas-f137" => "fas fa-chevron-circle-left",
			"fas-f138" => "fas fa-chevron-circle-right",
			"fas-f139" => "fas fa-chevron-circle-up",
			"fas-f078" => "fas fa-chevron-down",
			"fas-f053" => "fas fa-chevron-left",
			"fas-f054" => "fas fa-chevron-right",
			"fas-f077" => "fas fa-chevron-up",
			"fas-f1ae" => "fas fa-child",
			"fas-f51d" => "fas fa-church",
			"fas-f111" => "fas fa-circle",
			"fas-f1ce" => "fas fa-circle-notch",
			"fas-f64f" => "fas fa-city",
			"fas-f7f2" => "fas fa-clinic-medical",
			"fas-f328" => "fas fa-clipboard",
			"fas-f46c" => "fas fa-clipboard-check",
			"fas-f46d" => "fas fa-clipboard-list",
			"fas-f017" => "fas fa-clock",
			"fas-f24d" => "fas fa-clone",
			"fas-f20a" => "fas fa-closed-captioning",
			"fas-f0c2" => "fas fa-cloud",
			"fas-f381" => "fas fa-cloud-download-alt",
			"fas-f73b" => "fas fa-cloud-meatball",
			"fas-f6c3" => "fas fa-cloud-moon",
			"fas-f73c" => "fas fa-cloud-moon-rain",
			"fas-f73d" => "fas fa-cloud-rain",
			"fas-f740" => "fas fa-cloud-showers-heavy",
			"fas-f6c4" => "fas fa-cloud-sun",
			"fas-f743" => "fas fa-cloud-sun-rain",
			"fas-f382" => "fas fa-cloud-upload-alt",
			"fas-f561" => "fas fa-cocktail",
			"fas-f121" => "fas fa-code",
			"fas-f126" => "fas fa-code-branch",
			"fas-f0f4" => "fas fa-coffee",
			"fas-f013" => "fas fa-cog",
			"fas-f085" => "fas fa-cogs",
			"fas-f51e" => "fas fa-coins",
			"fas-f0db" => "fas fa-columns",
			"fas-f075" => "fas fa-comment",
			"fas-f27a" => "fas fa-comment-alt",
			"fas-f651" => "fas fa-comment-dollar",
			"fas-f4ad" => "fas fa-comment-dots",
			"fas-f7f5" => "fas fa-comment-medical",
			"fas-f4b3" => "fas fa-comment-slash",
			"fas-f086" => "fas fa-comments",
			"fas-f653" => "fas fa-comments-dollar",
			"fas-f51f" => "fas fa-compact-disc",
			"fas-f14e" => "fas fa-compass",
			"fas-f066" => "fas fa-compress",
			"fas-f422" => "fas fa-compress-alt",
			"fas-f78c" => "fas fa-compress-arrows-alt",
			"fas-f562" => "fas fa-concierge-bell",
			"fas-f563" => "fas fa-cookie",
			"fas-f564" => "fas fa-cookie-bite",
			"fas-f0c5" => "fas fa-copy",
			"fas-f1f9" => "fas fa-copyright",
			"fas-f4b8" => "fas fa-couch",
			"fas-f09d" => "fas fa-credit-card",
			"fas-f125" => "fas fa-crop",
			"fas-f565" => "fas fa-crop-alt",
			"fas-f654" => "fas fa-cross",
			"fas-f05b" => "fas fa-crosshairs",
			"fas-f520" => "fas fa-crow",
			"fas-f521" => "fas fa-crown",
			"fas-f7f7" => "fas fa-crutch",
			"fas-f1b2" => "fas fa-cube",
			"fas-f1b3" => "fas fa-cubes",
			"fas-f0c4" => "fas fa-cut",
			"fas-f1c0" => "fas fa-database",
			"fas-f2a4" => "fas fa-deaf",
			"fas-f747" => "fas fa-democrat",
			"fas-f108" => "fas fa-desktop",
			"fas-f655" => "fas fa-dharmachakra",
			"fas-f470" => "fas fa-diagnoses",
			"fas-f522" => "fas fa-dice",
			"fas-f6cf" => "fas fa-dice-d20",
			"fas-f6d1" => "fas fa-dice-d6",
			"fas-f523" => "fas fa-dice-five",
			"fas-f524" => "fas fa-dice-four",
			"fas-f525" => "fas fa-dice-one",
			"fas-f526" => "fas fa-dice-six",
			"fas-f527" => "fas fa-dice-three",
			"fas-f528" => "fas fa-dice-two",
			"fas-f566" => "fas fa-digital-tachograph",
			"fas-f5eb" => "fas fa-directions",
			"fas-f7fa" => "fas fa-disease",
			"fas-f529" => "fas fa-divide",
			"fas-f567" => "fas fa-dizzy",
			"fas-f471" => "fas fa-dna",
			"fas-f6d3" => "fas fa-dog",
			"fas-f155" => "fas fa-dollar-sign",
			"fas-f472" => "fas fa-dolly",
			"fas-f474" => "fas fa-dolly-flatbed",
			"fas-f4b9" => "fas fa-donate",
			"fas-f52a" => "fas fa-door-closed",
			"fas-f52b" => "fas fa-door-open",
			"fas-f192" => "fas fa-dot-circle",
			"fas-f4ba" => "fas fa-dove",
			"fas-f019" => "fas fa-download",
			"fas-f568" => "fas fa-drafting-compass",
			"fas-f6d5" => "fas fa-dragon",
			"fas-f5ee" => "fas fa-draw-polygon",
			"fas-f569" => "fas fa-drum",
			"fas-f56a" => "fas fa-drum-steelpan",
			"fas-f6d7" => "fas fa-drumstick-bite",
			"fas-f44b" => "fas fa-dumbbell",
			"fas-f793" => "fas fa-dumpster",
			"fas-f794" => "fas fa-dumpster-fire",
			"fas-f6d9" => "fas fa-dungeon",
			"fas-f044" => "fas fa-edit",
			"fas-f7fb" => "fas fa-egg",
			"fas-f052" => "fas fa-eject",
			"fas-f141" => "fas fa-ellipsis-h",
			"fas-f142" => "fas fa-ellipsis-v",
			"fas-f0e0" => "fas fa-envelope",
			"fas-f2b6" => "fas fa-envelope-open",
			"fas-f658" => "fas fa-envelope-open-text",
			"fas-f199" => "fas fa-envelope-square",
			"fas-f52c" => "fas fa-equals",
			"fas-f12d" => "fas fa-eraser",
			"fas-f796" => "fas fa-ethernet",
			"fas-f153" => "fas fa-euro-sign",
			"fas-f362" => "fas fa-exchange-alt",
			"fas-f12a" => "fas fa-exclamation",
			"fas-f06a" => "fas fa-exclamation-circle",
			"fas-f071" => "fas fa-exclamation-triangle",
			"fas-f065" => "fas fa-expand",
			"fas-f424" => "fas fa-expand-alt",
			"fas-f31e" => "fas fa-expand-arrows-alt",
			"fas-f35d" => "fas fa-external-link-alt",
			"fas-f360" => "fas fa-external-link-square-alt",
			"fas-f06e" => "fas fa-eye",
			"fas-f1fb" => "fas fa-eye-dropper",
			"fas-f070" => "fas fa-eye-slash",
			"fas-f863" => "fas fa-fan",
			"fas-f049" => "fas fa-fast-backward",
			"fas-f050" => "fas fa-fast-forward",
			"fas-f905" => "fas fa-faucet",
			"fas-f1ac" => "fas fa-fax",
			"fas-f52d" => "fas fa-feather",
			"fas-f56b" => "fas fa-feather-alt",
			"fas-f182" => "fas fa-female",
			"fas-f0fb" => "fas fa-fighter-jet",
			"fas-f15b" => "fas fa-file",
			"fas-f15c" => "fas fa-file-alt",
			"fas-f1c6" => "fas fa-file-archive",
			"fas-f1c7" => "fas fa-file-audio",
			"fas-f1c9" => "fas fa-file-code",
			"fas-f56c" => "fas fa-file-contract",
			"fas-f6dd" => "fas fa-file-csv",
			"fas-f56d" => "fas fa-file-download",
			"fas-f1c3" => "fas fa-file-excel",
			"fas-f56e" => "fas fa-file-export",
			"fas-f1c5" => "fas fa-file-image",
			"fas-f56f" => "fas fa-file-import",
			"fas-f570" => "fas fa-file-invoice",
			"fas-f571" => "fas fa-file-invoice-dollar",
			"fas-f477" => "fas fa-file-medical",
			"fas-f478" => "fas fa-file-medical-alt",
			"fas-f1c1" => "fas fa-file-pdf",
			"fas-f1c4" => "fas fa-file-powerpoint",
			"fas-f572" => "fas fa-file-prescription",
			"fas-f573" => "fas fa-file-signature",
			"fas-f574" => "fas fa-file-upload",
			"fas-f1c8" => "fas fa-file-video",
			"fas-f1c2" => "fas fa-file-word",
			"fas-f575" => "fas fa-fill",
			"fas-f576" => "fas fa-fill-drip",
			"fas-f008" => "fas fa-film",
			"fas-f0b0" => "fas fa-filter",
			"fas-f577" => "fas fa-fingerprint",
			"fas-f06d" => "fas fa-fire",
			"fas-f7e4" => "fas fa-fire-alt",
			"fas-f134" => "fas fa-fire-extinguisher",
			"fas-f479" => "fas fa-first-aid",
			"fas-f578" => "fas fa-fish",
			"fas-f6de" => "fas fa-fist-raised",
			"fas-f024" => "fas fa-flag",
			"fas-f11e" => "fas fa-flag-checkered",
			"fas-f74d" => "fas fa-flag-usa",
			"fas-f0c3" => "fas fa-flask",
			"fas-f579" => "fas fa-flushed",
			"fas-f07b" => "fas fa-folder",
			"fas-f65d" => "fas fa-folder-minus",
			"fas-f07c" => "fas fa-folder-open",
			"fas-f65e" => "fas fa-folder-plus",
			"fas-f031" => "fas fa-font",
			"fas-f44e" => "fas fa-football-ball",
			"fas-f04e" => "fas fa-forward",
			"fas-f52e" => "fas fa-frog",
			"fas-f119" => "fas fa-frown",
			"fas-f57a" => "fas fa-frown-open",
			"fas-f662" => "fas fa-funnel-dollar",
			"fas-f1e3" => "fas fa-futbol",
			"fas-f11b" => "fas fa-gamepad",
			"fas-f52f" => "fas fa-gas-pump",
			"fas-f0e3" => "fas fa-gavel",
			"fas-f3a5" => "fas fa-gem",
			"fas-f22d" => "fas fa-genderless",
			"fas-f6e2" => "fas fa-ghost",
			"fas-f06b" => "fas fa-gift",
			"fas-f79c" => "fas fa-gifts",
			"fas-f79f" => "fas fa-glass-cheers",
			"fas-f000" => "fas fa-glass-martini",
			"fas-f57b" => "fas fa-glass-martini-alt",
			"fas-f7a0" => "fas fa-glass-whiskey",
			"fas-f530" => "fas fa-glasses",
			"fas-f0ac" => "fas fa-globe",
			"fas-f57c" => "fas fa-globe-africa",
			"fas-f57d" => "fas fa-globe-americas",
			"fas-f57e" => "fas fa-globe-asia",
			"fas-f7a2" => "fas fa-globe-europe",
			"fas-f450" => "fas fa-golf-ball",
			"fas-f664" => "fas fa-gopuram",
			"fas-f19d" => "fas fa-graduation-cap",
			"fas-f531" => "fas fa-greater-than",
			"fas-f532" => "fas fa-greater-than-equal",
			"fas-f57f" => "fas fa-grimace",
			"fas-f580" => "fas fa-grin",
			"fas-f581" => "fas fa-grin-alt",
			"fas-f582" => "fas fa-grin-beam",
			"fas-f583" => "fas fa-grin-beam-sweat",
			"fas-f584" => "fas fa-grin-hearts",
			"fas-f585" => "fas fa-grin-squint",
			"fas-f586" => "fas fa-grin-squint-tears",
			"fas-f587" => "fas fa-grin-stars",
			"fas-f588" => "fas fa-grin-tears",
			"fas-f589" => "fas fa-grin-tongue",
			"fas-f58a" => "fas fa-grin-tongue-squint",
			"fas-f58b" => "fas fa-grin-tongue-wink",
			"fas-f58c" => "fas fa-grin-wink",
			"fas-f58d" => "fas fa-grip-horizontal",
			"fas-f7a4" => "fas fa-grip-lines",
			"fas-f7a5" => "fas fa-grip-lines-vertical",
			"fas-f58e" => "fas fa-grip-vertical",
			"fas-f7a6" => "fas fa-guitar",
			"fas-f0fd" => "fas fa-h-square",
			"fas-f805" => "fas fa-hamburger",
			"fas-f6e3" => "fas fa-hammer",
			"fas-f665" => "fas fa-hamsa",
			"fas-f4bd" => "fas fa-hand-holding",
			"fas-f4be" => "fas fa-hand-holding-heart",
			"fas-f95c" => "fas fa-hand-holding-medical",
			"fas-f4c0" => "fas fa-hand-holding-usd",
			"fas-f4c1" => "fas fa-hand-holding-water",
			"fas-f258" => "fas fa-hand-lizard",
			"fas-f806" => "fas fa-hand-middle-finger",
			"fas-f256" => "fas fa-hand-paper",
			"fas-f25b" => "fas fa-hand-peace",
			"fas-f0a7" => "fas fa-hand-point-down",
			"fas-f0a5" => "fas fa-hand-point-left",
			"fas-f0a4" => "fas fa-hand-point-right",
			"fas-f0a6" => "fas fa-hand-point-up",
			"fas-f25a" => "fas fa-hand-pointer",
			"fas-f255" => "fas fa-hand-rock",
			"fas-f257" => "fas fa-hand-scissors",
			"fas-f95d" => "fas fa-hand-sparkles",
			"fas-f259" => "fas fa-hand-spock",
			"fas-f4c2" => "fas fa-hands",
			"fas-f4c4" => "fas fa-hands-helping",
			"fas-f95e" => "fas fa-hands-wash",
			"fas-f2b5" => "fas fa-handshake",
			"fas-f95f" => "fas fa-handshake-alt-slash",
			"fas-f960" => "fas fa-handshake-slash",
			"fas-f6e6" => "fas fa-hanukiah",
			"fas-f807" => "fas fa-hard-hat",
			"fas-f292" => "fas fa-hashtag",
			"fas-f8c0" => "fas fa-hat-cowboy",
			"fas-f8c1" => "fas fa-hat-cowboy-side",
			"fas-f6e8" => "fas fa-hat-wizard",
			"fas-f0a0" => "fas fa-hdd",
			"fas-f961" => "fas fa-head-side-cough",
			"fas-f962" => "fas fa-head-side-cough-slash",
			"fas-f963" => "fas fa-head-side-mask",
			"fas-f964" => "fas fa-head-side-virus",
			"fas-f1dc" => "fas fa-heading",
			"fas-f025" => "fas fa-headphones",
			"fas-f58f" => "fas fa-headphones-alt",
			"fas-f590" => "fas fa-headset",
			"fas-f004" => "fas fa-heart",
			"fas-f7a9" => "fas fa-heart-broken",
			"fas-f21e" => "fas fa-heartbeat",
			"fas-f533" => "fas fa-helicopter",
			"fas-f591" => "fas fa-highlighter",
			"fas-f6ec" => "fas fa-hiking",
			"fas-f6ed" => "fas fa-hippo",
			"fas-f1da" => "fas fa-history",
			"fas-f453" => "fas fa-hockey-puck",
			"fas-f7aa" => "fas fa-holly-berry",
			"fas-f015" => "fas fa-home",
			"fas-f6f0" => "fas fa-horse",
			"fas-f7ab" => "fas fa-horse-head",
			"fas-f0f8" => "fas fa-hospital",
			"fas-f47d" => "fas fa-hospital-alt",
			"fas-f47e" => "fas fa-hospital-symbol",
			"fas-f80d" => "fas fa-hospital-user",
			"fas-f593" => "fas fa-hot-tub",
			"fas-f80f" => "fas fa-hotdog",
			"fas-f594" => "fas fa-hotel",
			"fas-f254" => "fas fa-hourglass",
			"fas-f253" => "fas fa-hourglass-end",
			"fas-f252" => "fas fa-hourglass-half",
			"fas-f251" => "fas fa-hourglass-start",
			"fas-f6f1" => "fas fa-house-damage",
			"fas-f965" => "fas fa-house-user",
			"fas-f6f2" => "fas fa-hryvnia",
			"fas-f246" => "fas fa-i-cursor",
			"fas-f810" => "fas fa-ice-cream",
			"fas-f7ad" => "fas fa-icicles",
			"fas-f86d" => "fas fa-icons",
			"fas-f2c1" => "fas fa-id-badge",
			"fas-f2c2" => "fas fa-id-card",
			"fas-f47f" => "fas fa-id-card-alt",
			"fas-f7ae" => "fas fa-igloo",
			"fas-f03e" => "fas fa-image",
			"fas-f302" => "fas fa-images",
			"fas-f01c" => "fas fa-inbox",
			"fas-f03c" => "fas fa-indent",
			"fas-f275" => "fas fa-industry",
			"fas-f534" => "fas fa-infinity",
			"fas-f129" => "fas fa-info",
			"fas-f05a" => "fas fa-info-circle",
			"fas-f033" => "fas fa-italic",
			"fas-f669" => "fas fa-jedi",
			"fas-f595" => "fas fa-joint",
			"fas-f66a" => "fas fa-journal-whills",
			"fas-f66b" => "fas fa-kaaba",
			"fas-f084" => "fas fa-key",
			"fas-f11c" => "fas fa-keyboard",
			"fas-f66d" => "fas fa-khanda",
			"fas-f596" => "fas fa-kiss",
			"fas-f597" => "fas fa-kiss-beam",
			"fas-f598" => "fas fa-kiss-wink-heart",
			"fas-f535" => "fas fa-kiwi-bird",
			"fas-f66f" => "fas fa-landmark",
			"fas-f1ab" => "fas fa-language",
			"fas-f109" => "fas fa-laptop",
			"fas-f5fc" => "fas fa-laptop-code",
			"fas-f966" => "fas fa-laptop-house",
			"fas-f812" => "fas fa-laptop-medical",
			"fas-f599" => "fas fa-laugh",
			"fas-f59a" => "fas fa-laugh-beam",
			"fas-f59b" => "fas fa-laugh-squint",
			"fas-f59c" => "fas fa-laugh-wink",
			"fas-f5fd" => "fas fa-layer-group",
			"fas-f06c" => "fas fa-leaf",
			"fas-f094" => "fas fa-lemon",
			"fas-f536" => "fas fa-less-than",
			"fas-f537" => "fas fa-less-than-equal",
			"fas-f3be" => "fas fa-level-down-alt",
			"fas-f3bf" => "fas fa-level-up-alt",
			"fas-f1cd" => "fas fa-life-ring",
			"fas-f0eb" => "fas fa-lightbulb",
			"fas-f0c1" => "fas fa-link",
			"fas-f195" => "fas fa-lira-sign",
			"fas-f03a" => "fas fa-list",
			"fas-f022" => "fas fa-list-alt",
			"fas-f0cb" => "fas fa-list-ol",
			"fas-f0ca" => "fas fa-list-ul",
			"fas-f124" => "fas fa-location-arrow",
			"fas-f023" => "fas fa-lock",
			"fas-f3c1" => "fas fa-lock-open",
			"fas-f309" => "fas fa-long-arrow-alt-down",
			"fas-f30a" => "fas fa-long-arrow-alt-left",
			"fas-f30b" => "fas fa-long-arrow-alt-right",
			"fas-f30c" => "fas fa-long-arrow-alt-up",
			"fas-f2a8" => "fas fa-low-vision",
			"fas-f59d" => "fas fa-luggage-cart",
			"fas-f604" => "fas fa-lungs",
			"fas-f967" => "fas fa-lungs-virus",
			"fas-f0d0" => "fas fa-magic",
			"fas-f076" => "fas fa-magnet",
			"fas-f674" => "fas fa-mail-bulk",
			"fas-f183" => "fas fa-male",
			"fas-f279" => "fas fa-map",
			"fas-f59f" => "fas fa-map-marked",
			"fas-f5a0" => "fas fa-map-marked-alt",
			"fas-f041" => "fas fa-map-marker",
			"fas-f3c5" => "fas fa-map-marker-alt",
			"fas-f276" => "fas fa-map-pin",
			"fas-f277" => "fas fa-map-signs",
			"fas-f5a1" => "fas fa-marker",
			"fas-f222" => "fas fa-mars",
			"fas-f227" => "fas fa-mars-double",
			"fas-f229" => "fas fa-mars-stroke",
			"fas-f22b" => "fas fa-mars-stroke-h",
			"fas-f22a" => "fas fa-mars-stroke-v",
			"fas-f6fa" => "fas fa-mask",
			"fas-f5a2" => "fas fa-medal",
			"fas-f0fa" => "fas fa-medkit",
			"fas-f11a" => "fas fa-meh",
			"fas-f5a4" => "fas fa-meh-blank",
			"fas-f5a5" => "fas fa-meh-rolling-eyes",
			"fas-f538" => "fas fa-memory",
			"fas-f676" => "fas fa-menorah",
			"fas-f223" => "fas fa-mercury",
			"fas-f753" => "fas fa-meteor",
			"fas-f2db" => "fas fa-microchip",
			"fas-f130" => "fas fa-microphone",
			"fas-f3c9" => "fas fa-microphone-alt",
			"fas-f539" => "fas fa-microphone-alt-slash",
			"fas-f131" => "fas fa-microphone-slash",
			"fas-f610" => "fas fa-microscope",
			"fas-f068" => "fas fa-minus",
			"fas-f056" => "fas fa-minus-circle",
			"fas-f146" => "fas fa-minus-square",
			"fas-f7b5" => "fas fa-mitten",
			"fas-f10b" => "fas fa-mobile",
			"fas-f3cd" => "fas fa-mobile-alt",
			"fas-f0d6" => "fas fa-money-bill",
			"fas-f3d1" => "fas fa-money-bill-alt",
			"fas-f53a" => "fas fa-money-bill-wave",
			"fas-f53b" => "fas fa-money-bill-wave-alt",
			"fas-f53c" => "fas fa-money-check",
			"fas-f53d" => "fas fa-money-check-alt",
			"fas-f5a6" => "fas fa-monument",
			"fas-f186" => "fas fa-moon",
			"fas-f5a7" => "fas fa-mortar-pestle",
			"fas-f678" => "fas fa-mosque",
			"fas-f21c" => "fas fa-motorcycle",
			"fas-f6fc" => "fas fa-mountain",
			"fas-f8cc" => "fas fa-mouse",
			"fas-f245" => "fas fa-mouse-pointer",
			"fas-f7b6" => "fas fa-mug-hot",
			"fas-f001" => "fas fa-music",
			"fas-f6ff" => "fas fa-network-wired",
			"fas-f22c" => "fas fa-neuter",
			"fas-f1ea" => "fas fa-newspaper",
			"fas-f53e" => "fas fa-not-equal",
			"fas-f481" => "fas fa-notes-medical",
			"fas-f247" => "fas fa-object-group",
			"fas-f248" => "fas fa-object-ungroup",
			"fas-f613" => "fas fa-oil-can",
			"fas-f679" => "fas fa-om",
			"fas-f700" => "fas fa-otter",
			"fas-f03b" => "fas fa-outdent",
			"fas-f815" => "fas fa-pager",
			"fas-f1fc" => "fas fa-paint-brush",
			"fas-f5aa" => "fas fa-paint-roller",
			"fas-f53f" => "fas fa-palette",
			"fas-f482" => "fas fa-pallet",
			"fas-f1d8" => "fas fa-paper-plane",
			"fas-f0c6" => "fas fa-paperclip",
			"fas-f4cd" => "fas fa-parachute-box",
			"fas-f1dd" => "fas fa-paragraph",
			"fas-f540" => "fas fa-parking",
			"fas-f5ab" => "fas fa-passport",
			"fas-f67b" => "fas fa-pastafarianism",
			"fas-f0ea" => "fas fa-paste",
			"fas-f04c" => "fas fa-pause",
			"fas-f28b" => "fas fa-pause-circle",
			"fas-f1b0" => "fas fa-paw",
			"fas-f67c" => "fas fa-peace",
			"fas-f304" => "fas fa-pen",
			"fas-f305" => "fas fa-pen-alt",
			"fas-f5ac" => "fas fa-pen-fancy",
			"fas-f5ad" => "fas fa-pen-nib",
			"fas-f14b" => "fas fa-pen-square",
			"fas-f303" => "fas fa-pencil-alt",
			"fas-f5ae" => "fas fa-pencil-ruler",
			"fas-f968" => "fas fa-people-arrows",
			"fas-f4ce" => "fas fa-people-carry",
			"fas-f816" => "fas fa-pepper-hot",
			"fas-f295" => "fas fa-percent",
			"fas-f541" => "fas fa-percentage",
			"fas-f756" => "fas fa-person-booth",
			"fas-f095" => "fas fa-phone",
			"fas-f879" => "fas fa-phone-alt",
			"fas-f3dd" => "fas fa-phone-slash",
			"fas-f098" => "fas fa-phone-square",
			"fas-f87b" => "fas fa-phone-square-alt",
			"fas-f2a0" => "fas fa-phone-volume",
			"fas-f87c" => "fas fa-photo-video",
			"fas-f4d3" => "fas fa-piggy-bank",
			"fas-f484" => "fas fa-pills",
			"fas-f818" => "fas fa-pizza-slice",
			"fas-f67f" => "fas fa-place-of-worship",
			"fas-f072" => "fas fa-plane",
			"fas-f5af" => "fas fa-plane-arrival",
			"fas-f5b0" => "fas fa-plane-departure",
			"fas-f969" => "fas fa-plane-slash",
			"fas-f04b" => "fas fa-play",
			"fas-f144" => "fas fa-play-circle",
			"fas-f1e6" => "fas fa-plug",
			"fas-f067" => "fas fa-plus",
			"fas-f055" => "fas fa-plus-circle",
			"fas-f0fe" => "fas fa-plus-square",
			"fas-f2ce" => "fas fa-podcast",
			"fas-f681" => "fas fa-poll",
			"fas-f682" => "fas fa-poll-h",
			"fas-f2fe" => "fas fa-poo",
			"fas-f75a" => "fas fa-poo-storm",
			"fas-f619" => "fas fa-poop",
			"fas-f3e0" => "fas fa-portrait",
			"fas-f154" => "fas fa-pound-sign",
			"fas-f011" => "fas fa-power-off",
			"fas-f683" => "fas fa-pray",
			"fas-f684" => "fas fa-praying-hands",
			"fas-f5b1" => "fas fa-prescription",
			"fas-f485" => "fas fa-prescription-bottle",
			"fas-f486" => "fas fa-prescription-bottle-alt",
			"fas-f02f" => "fas fa-print",
			"fas-f487" => "fas fa-procedures",
			"fas-f542" => "fas fa-project-diagram",
			"fas-f96a" => "fas fa-pump-medical",
			"fas-f96b" => "fas fa-pump-soap",
			"fas-f12e" => "fas fa-puzzle-piece",
			"fas-f029" => "fas fa-qrcode",
			"fas-f128" => "fas fa-question",
			"fas-f059" => "fas fa-question-circle",
			"fas-f458" => "fas fa-quidditch",
			"fas-f10d" => "fas fa-quote-left",
			"fas-f10e" => "fas fa-quote-right",
			"fas-f687" => "fas fa-quran",
			"fas-f7b9" => "fas fa-radiation",
			"fas-f7ba" => "fas fa-radiation-alt",
			"fas-f75b" => "fas fa-rainbow",
			"fas-f074" => "fas fa-random",
			"fas-f543" => "fas fa-receipt",
			"fas-f8d9" => "fas fa-record-vinyl",
			"fas-f1b8" => "fas fa-recycle",
			"fas-f01e" => "fas fa-redo",
			"fas-f2f9" => "fas fa-redo-alt",
			"fas-f25d" => "fas fa-registered",
			"fas-f87d" => "fas fa-remove-format",
			"fas-f3e5" => "fas fa-reply",
			"fas-f122" => "fas fa-reply-all",
			"fas-f75e" => "fas fa-republican",
			"fas-f7bd" => "fas fa-restroom",
			"fas-f079" => "fas fa-retweet",
			"fas-f4d6" => "fas fa-ribbon",
			"fas-f70b" => "fas fa-ring",
			"fas-f018" => "fas fa-road",
			"fas-f544" => "fas fa-robot",
			"fas-f135" => "fas fa-rocket",
			"fas-f4d7" => "fas fa-route",
			"fas-f09e" => "fas fa-rss",
			"fas-f143" => "fas fa-rss-square",
			"fas-f158" => "fas fa-ruble-sign",
			"fas-f545" => "fas fa-ruler",
			"fas-f546" => "fas fa-ruler-combined",
			"fas-f547" => "fas fa-ruler-horizontal",
			"fas-f548" => "fas fa-ruler-vertical",
			"fas-f70c" => "fas fa-running",
			"fas-f156" => "fas fa-rupee-sign",
			"fas-f5b3" => "fas fa-sad-cry",
			"fas-f5b4" => "fas fa-sad-tear",
			"fas-f7bf" => "fas fa-satellite",
			"fas-f7c0" => "fas fa-satellite-dish",
			"fas-f0c7" => "fas fa-save",
			"fas-f549" => "fas fa-school",
			"fas-f54a" => "fas fa-screwdriver",
			"fas-f70e" => "fas fa-scroll",
			"fas-f7c2" => "fas fa-sd-card",
			"fas-f002" => "fas fa-search",
			"fas-f688" => "fas fa-search-dollar",
			"fas-f689" => "fas fa-search-location",
			"fas-f010" => "fas fa-search-minus",
			"fas-f00e" => "fas fa-search-plus",
			"fas-f4d8" => "fas fa-seedling",
			"fas-f233" => "fas fa-server",
			"fas-f61f" => "fas fa-shapes",
			"fas-f064" => "fas fa-share",
			"fas-f1e0" => "fas fa-share-alt",
			"fas-f1e1" => "fas fa-share-alt-square",
			"fas-f14d" => "fas fa-share-square",
			"fas-f20b" => "fas fa-shekel-sign",
			"fas-f3ed" => "fas fa-shield-alt",
			"fas-f96c" => "fas fa-shield-virus",
			"fas-f21a" => "fas fa-ship",
			"fas-f48b" => "fas fa-shipping-fast",
			"fas-f54b" => "fas fa-shoe-prints",
			"fas-f290" => "fas fa-shopping-bag",
			"fas-f291" => "fas fa-shopping-basket",
			"fas-f07a" => "fas fa-shopping-cart",
			"fas-f2cc" => "fas fa-shower",
			"fas-f5b6" => "fas fa-shuttle-van",
			"fas-f4d9" => "fas fa-sign",
			"fas-f2f6" => "fas fa-sign-in-alt",
			"fas-f2a7" => "fas fa-sign-language",
			"fas-f2f5" => "fas fa-sign-out-alt",
			"fas-f012" => "fas fa-signal",
			"fas-f5b7" => "fas fa-signature",
			"fas-f7c4" => "fas fa-sim-card",
			"fas-f0e8" => "fas fa-sitemap",
			"fas-f7c5" => "fas fa-skating",
			"fas-f7c9" => "fas fa-skiing",
			"fas-f7ca" => "fas fa-skiing-nordic",
			"fas-f54c" => "fas fa-skull",
			"fas-f714" => "fas fa-skull-crossbones",
			"fas-f715" => "fas fa-slash",
			"fas-f7cc" => "fas fa-sleigh",
			"fas-f1de" => "fas fa-sliders-h",
			"fas-f118" => "fas fa-smile",
			"fas-f5b8" => "fas fa-smile-beam",
			"fas-f4da" => "fas fa-smile-wink",
			"fas-f75f" => "fas fa-smog",
			"fas-f48d" => "fas fa-smoking",
			"fas-f54d" => "fas fa-smoking-ban",
			"fas-f7cd" => "fas fa-sms",
			"fas-f7ce" => "fas fa-snowboarding",
			"fas-f2dc" => "fas fa-snowflake",
			"fas-f7d0" => "fas fa-snowman",
			"fas-f7d2" => "fas fa-snowplow",
			"fas-f96e" => "fas fa-soap",
			"fas-f696" => "fas fa-socks",
			"fas-f5ba" => "fas fa-solar-panel",
			"fas-f0dc" => "fas fa-sort",
			"fas-f15d" => "fas fa-sort-alpha-down",
			"fas-f881" => "fas fa-sort-alpha-down-alt",
			"fas-f15e" => "fas fa-sort-alpha-up",
			"fas-f882" => "fas fa-sort-alpha-up-alt",
			"fas-f160" => "fas fa-sort-amount-down",
			"fas-f884" => "fas fa-sort-amount-down-alt",
			"fas-f161" => "fas fa-sort-amount-up",
			"fas-f885" => "fas fa-sort-amount-up-alt",
			"fas-f0dd" => "fas fa-sort-down",
			"fas-f162" => "fas fa-sort-numeric-down",
			"fas-f886" => "fas fa-sort-numeric-down-alt",
			"fas-f163" => "fas fa-sort-numeric-up",
			"fas-f887" => "fas fa-sort-numeric-up-alt",
			"fas-f0de" => "fas fa-sort-up",
			"fas-f5bb" => "fas fa-spa",
			"fas-f197" => "fas fa-space-shuttle",
			"fas-f891" => "fas fa-spell-check",
			"fas-f717" => "fas fa-spider",
			"fas-f110" => "fas fa-spinner",
			"fas-f5bc" => "fas fa-splotch",
			"fas-f5bd" => "fas fa-spray-can",
			"fas-f0c8" => "fas fa-square",
			"fas-f45c" => "fas fa-square-full",
			"fas-f698" => "fas fa-square-root-alt",
			"fas-f5bf" => "fas fa-stamp",
			"fas-f005" => "fas fa-star",
			"fas-f699" => "fas fa-star-and-crescent",
			"fas-f089" => "fas fa-star-half",
			"fas-f5c0" => "fas fa-star-half-alt",
			"fas-f69a" => "fas fa-star-of-david",
			"fas-f621" => "fas fa-star-of-life",
			"fas-f048" => "fas fa-step-backward",
			"fas-f051" => "fas fa-step-forward",
			"fas-f0f1" => "fas fa-stethoscope",
			"fas-f249" => "fas fa-sticky-note",
			"fas-f04d" => "fas fa-stop",
			"fas-f28d" => "fas fa-stop-circle",
			"fas-f2f2" => "fas fa-stopwatch",
			"fas-f96f" => "fas fa-stopwatch-20",
			"fas-f54e" => "fas fa-store",
			"fas-f54f" => "fas fa-store-alt",
			"fas-f970" => "fas fa-store-alt-slash",
			"fas-f971" => "fas fa-store-slash",
			"fas-f550" => "fas fa-stream",
			"fas-f21d" => "fas fa-street-view",
			"fas-f0cc" => "fas fa-strikethrough",
			"fas-f551" => "fas fa-stroopwafel",
			"fas-f12c" => "fas fa-subscript",
			"fas-f239" => "fas fa-subway",
			"fas-f0f2" => "fas fa-suitcase",
			"fas-f5c1" => "fas fa-suitcase-rolling",
			"fas-f185" => "fas fa-sun",
			"fas-f12b" => "fas fa-superscript",
			"fas-f5c2" => "fas fa-surprise",
			"fas-f5c3" => "fas fa-swatchbook",
			"fas-f5c4" => "fas fa-swimmer",
			"fas-f5c5" => "fas fa-swimming-pool",
			"fas-f69b" => "fas fa-synagogue",
			"fas-f021" => "fas fa-sync",
			"fas-f2f1" => "fas fa-sync-alt",
			"fas-f48e" => "fas fa-syringe",
			"fas-f0ce" => "fas fa-table",
			"fas-f45d" => "fas fa-table-tennis",
			"fas-f10a" => "fas fa-tablet",
			"fas-f3fa" => "fas fa-tablet-alt",
			"fas-f490" => "fas fa-tablets",
			"fas-f3fd" => "fas fa-tachometer-alt",
			"fas-f02b" => "fas fa-tag",
			"fas-f02c" => "fas fa-tags",
			"fas-f4db" => "fas fa-tape",
			"fas-f0ae" => "fas fa-tasks",
			"fas-f1ba" => "fas fa-taxi",
			"fas-f62e" => "fas fa-teeth",
			"fas-f62f" => "fas fa-teeth-open",
			"fas-f769" => "fas fa-temperature-high",
			"fas-f76b" => "fas fa-temperature-low",
			"fas-f7d7" => "fas fa-tenge",
			"fas-f120" => "fas fa-terminal",
			"fas-f034" => "fas fa-text-height",
			"fas-f035" => "fas fa-text-width",
			"fas-f00a" => "fas fa-th",
			"fas-f009" => "fas fa-th-large",
			"fas-f00b" => "fas fa-th-list",
			"fas-f630" => "fas fa-theater-masks",
			"fas-f491" => "fas fa-thermometer",
			"fas-f2cb" => "fas fa-thermometer-empty",
			"fas-f2c7" => "fas fa-thermometer-full",
			"fas-f2c9" => "fas fa-thermometer-half",
			"fas-f2ca" => "fas fa-thermometer-quarter",
			"fas-f2c8" => "fas fa-thermometer-three-quarters",
			"fas-f165" => "fas fa-thumbs-down",
			"fas-f164" => "fas fa-thumbs-up",
			"fas-f08d" => "fas fa-thumbtack",
			"fas-f3ff" => "fas fa-ticket-alt",
			"fas-f00d" => "fas fa-times",
			"fas-f057" => "fas fa-times-circle",
			"fas-f043" => "fas fa-tint",
			"fas-f5c7" => "fas fa-tint-slash",
			"fas-f5c8" => "fas fa-tired",
			"fas-f204" => "fas fa-toggle-off",
			"fas-f205" => "fas fa-toggle-on",
			"fas-f7d8" => "fas fa-toilet",
			"fas-f71e" => "fas fa-toilet-paper",
			"fas-f972" => "fas fa-toilet-paper-slash",
			"fas-f552" => "fas fa-toolbox",
			"fas-f7d9" => "fas fa-tools",
			"fas-f5c9" => "fas fa-tooth",
			"fas-f6a0" => "fas fa-torah",
			"fas-f6a1" => "fas fa-torii-gate",
			"fas-f722" => "fas fa-tractor",
			"fas-f25c" => "fas fa-trademark",
			"fas-f637" => "fas fa-traffic-light",
			"fas-f941" => "fas fa-trailer",
			"fas-f238" => "fas fa-train",
			"fas-f7da" => "fas fa-tram",
			"fas-f224" => "fas fa-transgender",
			"fas-f225" => "fas fa-transgender-alt",
			"fas-f1f8" => "fas fa-trash",
			"fas-f2ed" => "fas fa-trash-alt",
			"fas-f829" => "fas fa-trash-restore",
			"fas-f82a" => "fas fa-trash-restore-alt",
			"fas-f1bb" => "fas fa-tree",
			"fas-f091" => "fas fa-trophy",
			"fas-f0d1" => "fas fa-truck",
			"fas-f4de" => "fas fa-truck-loading",
			"fas-f63b" => "fas fa-truck-monster",
			"fas-f4df" => "fas fa-truck-moving",
			"fas-f63c" => "fas fa-truck-pickup",
			"fas-f553" => "fas fa-tshirt",
			"fas-f1e4" => "fas fa-tty",
			"fas-f26c" => "fas fa-tv",
			"fas-f0e9" => "fas fa-umbrella",
			"fas-f5ca" => "fas fa-umbrella-beach",
			"fas-f0cd" => "fas fa-underline",
			"fas-f0e2" => "fas fa-undo",
			"fas-f2ea" => "fas fa-undo-alt",
			"fas-f29a" => "fas fa-universal-access",
			"fas-f19c" => "fas fa-university",
			"fas-f127" => "fas fa-unlink",
			"fas-f09c" => "fas fa-unlock",
			"fas-f13e" => "fas fa-unlock-alt",
			"fas-f093" => "fas fa-upload",
			"fas-f007" => "fas fa-user",
			"fas-f406" => "fas fa-user-alt",
			"fas-f4fa" => "fas fa-user-alt-slash",
			"fas-f4fb" => "fas fa-user-astronaut",
			"fas-f4fc" => "fas fa-user-check",
			"fas-f2bd" => "fas fa-user-circle",
			"fas-f4fd" => "fas fa-user-clock",
			"fas-f4fe" => "fas fa-user-cog",
			"fas-f4ff" => "fas fa-user-edit",
			"fas-f500" => "fas fa-user-friends",
			"fas-f501" => "fas fa-user-graduate",
			"fas-f728" => "fas fa-user-injured",
			"fas-f502" => "fas fa-user-lock",
			"fas-f0f0" => "fas fa-user-md",
			"fas-f503" => "fas fa-user-minus",
			"fas-f504" => "fas fa-user-ninja",
			"fas-f82f" => "fas fa-user-nurse",
			"fas-f234" => "fas fa-user-plus",
			"fas-f21b" => "fas fa-user-secret",
			"fas-f505" => "fas fa-user-shield",
			"fas-f506" => "fas fa-user-slash",
			"fas-f507" => "fas fa-user-tag",
			"fas-f508" => "fas fa-user-tie",
			"fas-f235" => "fas fa-user-times",
			"fas-f0c0" => "fas fa-users",
			"fas-f509" => "fas fa-users-cog",
			"fas-f2e5" => "fas fa-utensil-spoon",
			"fas-f2e7" => "fas fa-utensils",
			"fas-f5cb" => "fas fa-vector-square",
			"fas-f221" => "fas fa-venus",
			"fas-f226" => "fas fa-venus-double",
			"fas-f228" => "fas fa-venus-mars",
			"fas-f492" => "fas fa-vial",
			"fas-f493" => "fas fa-vials",
			"fas-f03d" => "fas fa-video",
			"fas-f4e2" => "fas fa-video-slash",
			"fas-f6a7" => "fas fa-vihara",
			"fas-f974" => "fas fa-virus",
			"fas-f975" => "fas fa-virus-slash",
			"fas-f976" => "fas fa-viruses",
			"fas-f897" => "fas fa-voicemail",
			"fas-f45f" => "fas fa-volleyball-ball",
			"fas-f027" => "fas fa-volume-down",
			"fas-f6a9" => "fas fa-volume-mute",
			"fas-f026" => "fas fa-volume-off",
			"fas-f028" => "fas fa-volume-up",
			"fas-f772" => "fas fa-vote-yea",
			"fas-f729" => "fas fa-vr-cardboard",
			"fas-f554" => "fas fa-walking",
			"fas-f555" => "fas fa-wallet",
			"fas-f494" => "fas fa-warehouse",
			"fas-f773" => "fas fa-water",
			"fas-f83e" => "fas fa-wave-square",
			"fas-f496" => "fas fa-weight",
			"fas-f5cd" => "fas fa-weight-hanging",
			"fas-f193" => "fas fa-wheelchair",
			"fas-f1eb" => "fas fa-wifi",
			"fas-f72e" => "fas fa-wind",
			"fas-f410" => "fas fa-window-close",
			"fas-f2d0" => "fas fa-window-maximize",
			"fas-f2d1" => "fas fa-window-minimize",
			"fas-f2d2" => "fas fa-window-restore",
			"fas-f72f" => "fas fa-wine-bottle",
			"fas-f4e3" => "fas fa-wine-glass",
			"fas-f5ce" => "fas fa-wine-glass-alt",
			"fas-f159" => "fas fa-won-sign",
			"fas-f0ad" => "fas fa-wrench",
			"fas-f497" => "fas fa-x-ray",
			"fas-f157" => "fas fa-yen-sign",
			"fas-f6ad" => "fas fa-yin-y"
		);

		$regular_icons = array(	
			"far-f2b9" => "far fa-address-book",
			"far-f2bb" => "far fa-address-card",
			"far-f556" => "far fa-angry",
			"far-f358" => "far fa-arrow-alt-circle-down",
			"far-f359" => "far fa-arrow-alt-circle-left",
			"far-f35a" => "far fa-arrow-alt-circle-right",
			"far-f35b" => "far fa-arrow-alt-circle-up",
			"far-f0f3" => "far fa-bell",
			"far-f1f6" => "far fa-bell-slash",
			"far-f02e" => "far fa-bookmark",
			"far-f1ad" => "far fa-building",
			"far-f133" => "far fa-calendar",
			"far-f073" => "far fa-calendar-alt",
			"far-f274" => "far fa-calendar-check",
			"far-f272" => "far fa-calendar-minus",
			"far-f271" => "far fa-calendar-plus",
			"far-f273" => "far fa-calendar-times",
			"far-f150" => "far fa-caret-square-down",
			"far-f191" => "far fa-caret-square-left",
			"far-f152" => "far fa-caret-square-right",
			"far-f151" => "far fa-caret-square-up",
			"far-f080" => "far fa-chart-bar",
			"far-f058" => "far fa-check-circle",
			"far-f14a" => "far fa-check-square",
			"far-f111" => "far fa-circle",
			"far-f328" => "far fa-clipboard",
			"far-f017" => "far fa-clock",
			"far-f24d" => "far fa-clone",
			"far-f20a" => "far fa-closed-captioning",
			"far-f075" => "far fa-comment",
			"far-f27a" => "far fa-comment-alt",
			"far-f4ad" => "far fa-comment-dots",
			"far-f086" => "far fa-comments",
			"far-f14e" => "far fa-compass",
			"far-f0c5" => "far fa-copy",
			"far-f1f9" => "far fa-copyright",
			"far-f09d" => "far fa-credit-card",
			"far-f567" => "far fa-dizzy",
			"far-f192" => "far fa-dot-circle",
			"far-f044" => "far fa-edit",
			"far-f0e0" => "far fa-envelope",
			"far-f2b6" => "far fa-envelope-open",
			"far-f06e" => "far fa-eye",
			"far-f070" => "far fa-eye-slash",
			"far-f15b" => "far fa-file",
			"far-f15c" => "far fa-file-alt",
			"far-f1c6" => "far fa-file-archive",
			"far-f1c7" => "far fa-file-audio",
			"far-f1c9" => "far fa-file-code",
			"far-f1c3" => "far fa-file-excel",
			"far-f1c5" => "far fa-file-image",
			"far-f1c1" => "far fa-file-pdf",
			"far-f1c4" => "far fa-file-powerpoint",
			"far-f1c8" => "far fa-file-video",
			"far-f1c2" => "far fa-file-word",
			"far-f024" => "far fa-flag",
			"far-f579" => "far fa-flushed",
			"far-f07b" => "far fa-folder",
			"far-f07c" => "far fa-folder-open",
			"far-f119" => "far fa-frown",
			"far-f57a" => "far fa-frown-open",
			"far-f1e3" => "far fa-futbol",
			"far-f3a5" => "far fa-gem",
			"far-f57f" => "far fa-grimace",
			"far-f580" => "far fa-grin",
			"far-f581" => "far fa-grin-alt",
			"far-f582" => "far fa-grin-beam",
			"far-f583" => "far fa-grin-beam-sweat",
			"far-f584" => "far fa-grin-hearts",
			"far-f585" => "far fa-grin-squint",
			"far-f586" => "far fa-grin-squint-tears",
			"far-f587" => "far fa-grin-stars",
			"far-f588" => "far fa-grin-tears",
			"far-f589" => "far fa-grin-tongue",
			"far-f58a" => "far fa-grin-tongue-squint",
			"far-f58b" => "far fa-grin-tongue-wink",
			"far-f58c" => "far fa-grin-wink",
			"far-f258" => "far fa-hand-lizard",
			"far-f256" => "far fa-hand-paper",
			"far-f25b" => "far fa-hand-peace",
			"far-f0a7" => "far fa-hand-point-down",
			"far-f0a5" => "far fa-hand-point-left",
			"far-f0a4" => "far fa-hand-point-right",
			"far-f0a6" => "far fa-hand-point-up",
			"far-f25a" => "far fa-hand-pointer",
			"far-f255" => "far fa-hand-rock",
			"far-f257" => "far fa-hand-scissors",
			"far-f259" => "far fa-hand-spock",
			"far-f2b5" => "far fa-handshake",
			"far-f0a0" => "far fa-hdd",
			"far-f004" => "far fa-heart",
			"far-f0f8" => "far fa-hospital",
			"far-f254" => "far fa-hourglass",
			"far-f2c1" => "far fa-id-badge",
			"far-f2c2" => "far fa-id-card",
			"far-f03e" => "far fa-image",
			"far-f302" => "far fa-images",
			"far-f11c" => "far fa-keyboard",
			"far-f596" => "far fa-kiss",
			"far-f597" => "far fa-kiss-beam",
			"far-f598" => "far fa-kiss-wink-heart",
			"far-f599" => "far fa-laugh",
			"far-f59a" => "far fa-laugh-beam",
			"far-f59b" => "far fa-laugh-squint",
			"far-f59c" => "far fa-laugh-wink",
			"far-f094" => "far fa-lemon",
			"far-f1cd" => "far fa-life-ring",
			"far-f0eb" => "far fa-lightbulb",
			"far-f022" => "far fa-list-alt",
			"far-f279" => "far fa-map",
			"far-f11a" => "far fa-meh",
			"far-f5a4" => "far fa-meh-blank",
			"far-f5a5" => "far fa-meh-rolling-eyes",
			"far-f146" => "far fa-minus-square",
			"far-f3d1" => "far fa-money-bill-alt",
			"far-f186" => "far fa-moon",
			"far-f1ea" => "far fa-newspaper",
			"far-f247" => "far fa-object-group",
			"far-f248" => "far fa-object-ungroup",
			"far-f1d8" => "far fa-paper-plane",
			"far-f28b" => "far fa-pause-circle",
			"far-f144" => "far fa-play-circle",
			"far-f0fe" => "far fa-plus-square",
			"far-f059" => "far fa-question-circle",
			"far-f25d" => "far fa-registered",
			"far-f5b3" => "far fa-sad-cry",
			"far-f5b4" => "far fa-sad-tear",
			"far-f0c7" => "far fa-save",
			"far-f14d" => "far fa-share-square",
			"far-f118" => "far fa-smile",
			"far-f5b8" => "far fa-smile-beam",
			"far-f4da" => "far fa-smile-wink",
			"far-f2dc" => "far fa-snowflake",
			"far-f0c8" => "far fa-square",
			"far-f005" => "far fa-star",
			"far-f089" => "far fa-star-half",
			"far-f249" => "far fa-sticky-note",
			"far-f28d" => "far fa-stop-circle",
			"far-f185" => "far fa-sun",
			"far-f5c2" => "far fa-surprise",
			"far-f165" => "far fa-thumbs-down",
			"far-f164" => "far fa-thumbs-up",
			"far-f057" => "far fa-times-circle",
			"far-f5c8" => "far fa-tired",
			"far-f2ed" => "far fa-trash-alt",
			"far-f007" => "far fa-user",
			"far-f2bd" => "far fa-user-circle",
			"far-f410" => "far fa-window-close",
			"far-f2d0" => "far fa-window-maximize",
			"far-f2d1" => "far fa-window-minimize",
			"far-f2d2" => "far fa-window-rest"
		);

		$brand_icons = array(
			"fab-f26e" => "fab fa-500px",
			"fab-f368" => "fab fa-accessible-icon",
			"fab-f369" => "fab fa-accusoft",
			"fab-f6af" => "fab fa-acquisitions-incorporated",
			"fab-f170" => "fab fa-adn",
			"fab-f778" => "fab fa-adobe",
			"fab-f36a" => "fab fa-adversal",
			"fab-f36b" => "fab fa-affiliatetheme",
			"fab-f834" => "fab fa-airbnb",
			"fab-f36c" => "fab fa-algolia",
			"fab-f642" => "fab fa-alipay",
			"fab-f270" => "fab fa-amazon",
			"fab-f42c" => "fab fa-amazon-pay",
			"fab-f36d" => "fab fa-amilia",
			"fab-f17b" => "fab fa-android",
			"fab-f209" => "fab fa-angellist",
			"fab-f36e" => "fab fa-angrycreative",
			"fab-f420" => "fab fa-angular",
			"fab-f36f" => "fab fa-app-store",
			"fab-f370" => "fab fa-app-store-ios",
			"fab-f371" => "fab fa-apper",
			"fab-f179" => "fab fa-apple",
			"fab-f415" => "fab fa-apple-pay",
			"fab-f77a" => "fab fa-artstation",
			"fab-f372" => "fab fa-asymmetrik",
			"fab-f77b" => "fab fa-atlassian",
			"fab-f373" => "fab fa-audible",
			"fab-f41c" => "fab fa-autoprefixer",
			"fab-f374" => "fab fa-avianex",
			"fab-f421" => "fab fa-aviato",
			"fab-f375" => "fab fa-aws",
			"fab-f2d5" => "fab fa-bandcamp",
			"fab-f835" => "fab fa-battle-net",
			"fab-f1b4" => "fab fa-behance",
			"fab-f1b5" => "fab fa-behance-square",
			"fab-f378" => "fab fa-bimobject",
			"fab-f171" => "fab fa-bitbucket",
			"fab-f379" => "fab fa-bitcoin",
			"fab-f37a" => "fab fa-bity",
			"fab-f27e" => "fab fa-black-tie",
			"fab-f37b" => "fab fa-blackberry",
			"fab-f37c" => "fab fa-blogger",
			"fab-f37d" => "fab fa-blogger-b",
			"fab-f293" => "fab fa-bluetooth",
			"fab-f294" => "fab fa-bluetooth-b",
			"fab-f836" => "fab fa-bootstrap",
			"fab-f15a" => "fab fa-btc",
			"fab-f837" => "fab fa-buffer",
			"fab-f37f" => "fab fa-buromobelexperte",
			"fab-f8a6" => "fab fa-buy-n-large",
			"fab-f20d" => "fab fa-buysellads",
			"fab-f785" => "fab fa-canadian-maple-leaf",
			"fab-f42d" => "fab fa-cc-amazon-pay",
			"fab-f1f3" => "fab fa-cc-amex",
			"fab-f416" => "fab fa-cc-apple-pay",
			"fab-f24c" => "fab fa-cc-diners-club",
			"fab-f1f2" => "fab fa-cc-discover",
			"fab-f24b" => "fab fa-cc-jcb",
			"fab-f1f1" => "fab fa-cc-mastercard",
			"fab-f1f4" => "fab fa-cc-paypal",
			"fab-f1f5" => "fab fa-cc-stripe",
			"fab-f1f0" => "fab fa-cc-visa",
			"fab-f380" => "fab fa-centercode",
			"fab-f789" => "fab fa-centos",
			"fab-f268" => "fab fa-chrome",
			"fab-f838" => "fab fa-chromecast",
			"fab-f383" => "fab fa-cloudscale",
			"fab-f384" => "fab fa-cloudsmith",
			"fab-f385" => "fab fa-cloudversify",
			"fab-f1cb" => "fab fa-codepen",
			"fab-f284" => "fab fa-codiepie",
			"fab-f78d" => "fab fa-confluence",
			"fab-f20e" => "fab fa-connectdevelop",
			"fab-f26d" => "fab fa-contao",
			"fab-f89e" => "fab fa-cotton-bureau",
			"fab-f388" => "fab fa-cpanel",
			"fab-f25e" => "fab fa-creative-commons",
			"fab-f4e7" => "fab fa-creative-commons-by",
			"fab-f4e8" => "fab fa-creative-commons-nc",
			"fab-f4e9" => "fab fa-creative-commons-nc-eu",
			"fab-f4ea" => "fab fa-creative-commons-nc-jp",
			"fab-f4eb" => "fab fa-creative-commons-nd",
			"fab-f4ec" => "fab fa-creative-commons-pd",
			"fab-f4ed" => "fab fa-creative-commons-pd-alt",
			"fab-f4ee" => "fab fa-creative-commons-remix",
			"fab-f4ef" => "fab fa-creative-commons-sa",
			"fab-f4f0" => "fab fa-creative-commons-sampling",
			"fab-f4f1" => "fab fa-creative-commons-sampling-plus",
			"fab-f4f2" => "fab fa-creative-commons-share",
			"fab-f4f3" => "fab fa-creative-commons-zero",
			"fab-f6c9" => "fab fa-critical-role",
			"fab-f13c" => "fab fa-css3",
			"fab-f38b" => "fab fa-css3-alt",
			"fab-f38c" => "fab fa-cuttlefish",
			"fab-f38d" => "fab fa-d-and-d",
			"fab-f6ca" => "fab fa-d-and-d-beyond",
			"fab-f952" => "fab fa-dailymotion",
			"fab-f210" => "fab fa-dashcube",
			"fab-f1a5" => "fab fa-delicious",
			"fab-f38e" => "fab fa-deploydog",
			"fab-f38f" => "fab fa-deskpro",
			"fab-f6cc" => "fab fa-dev",
			"fab-f1bd" => "fab fa-deviantart",
			"fab-f790" => "fab fa-dhl",
			"fab-f791" => "fab fa-diaspora",
			"fab-f1a6" => "fab fa-digg",
			"fab-f391" => "fab fa-digital-ocean",
			"fab-f392" => "fab fa-discord",
			"fab-f393" => "fab fa-discourse",
			"fab-f394" => "fab fa-dochub",
			"fab-f395" => "fab fa-docker",
			"fab-f396" => "fab fa-draft2digital",
			"fab-f17d" => "fab fa-dribbble",
			"fab-f397" => "fab fa-dribbble-square",
			"fab-f16b" => "fab fa-dropbox",
			"fab-f1a9" => "fab fa-drupal",
			"fab-f399" => "fab fa-dyalog",
			"fab-f39a" => "fab fa-earlybirds",
			"fab-f4f4" => "fab fa-ebay",
			"fab-f282" => "fab fa-edge",
			"fab-f430" => "fab fa-elementor",
			"fab-f5f1" => "fab fa-ello",
			"fab-f423" => "fab fa-ember",
			"fab-f1d1" => "fab fa-empire",
			"fab-f299" => "fab fa-envira",
			"fab-f39d" => "fab fa-erlang",
			"fab-f42e" => "fab fa-ethereum",
			"fab-f2d7" => "fab fa-etsy",
			"fab-f839" => "fab fa-evernote",
			"fab-f23e" => "fab fa-expeditedssl",
			"fab-f09a" => "fab fa-facebook",
			"fab-f39e" => "fab fa-facebook-f",
			"fab-f39f" => "fab fa-facebook-messenger",
			"fab-f082" => "fab fa-facebook-square",
			"fab-f6dc" => "fab fa-fantasy-flight-games",
			"fab-f797" => "fab fa-fedex",
			"fab-f798" => "fab fa-fedora",
			"fab-f799" => "fab fa-figma",
			"fab-f269" => "fab fa-firefox",
			"fab-f907" => "fab fa-firefox-browser",
			"fab-f2b0" => "fab fa-first-order",
			"fab-f50a" => "fab fa-first-order-alt",
			"fab-f3a1" => "fab fa-firstdraft",
			"fab-f16e" => "fab fa-flickr",
			"fab-f44d" => "fab fa-flipboard",
			"fab-f417" => "fab fa-fly",
			"fab-f2b4" => "fab fa-font-awesome",
			"fab-f35c" => "fab fa-font-awesome-alt",
			"fab-f425" => "fab fa-font-awesome-flag",
			"fab-f280" => "fab fa-fonticons",
			"fab-f3a2" => "fab fa-fonticons-fi",
			"fab-f286" => "fab fa-fort-awesome",
			"fab-f3a3" => "fab fa-fort-awesome-alt",
			"fab-f211" => "fab fa-forumbee",
			"fab-f180" => "fab fa-foursquare",
			"fab-f2c5" => "fab fa-free-code-camp",
			"fab-f3a4" => "fab fa-freebsd",
			"fab-f50b" => "fab fa-fulcrum",
			"fab-f50c" => "fab fa-galactic-republic",
			"fab-f50d" => "fab fa-galactic-senate",
			"fab-f265" => "fab fa-get-pocket",
			"fab-f260" => "fab fa-gg",
			"fab-f261" => "fab fa-gg-circle",
			"fab-f1d3" => "fab fa-git",
			"fab-f841" => "fab fa-git-alt",
			"fab-f1d2" => "fab fa-git-square",
			"fab-f09b" => "fab fa-github",
			"fab-f113" => "fab fa-github-alt",
			"fab-f092" => "fab fa-github-square",
			"fab-f3a6" => "fab fa-gitkraken",
			"fab-f296" => "fab fa-gitlab",
			"fab-f426" => "fab fa-gitter",
			"fab-f2a5" => "fab fa-glide",
			"fab-f2a6" => "fab fa-glide-g",
			"fab-f3a7" => "fab fa-gofore",
			"fab-f3a8" => "fab fa-goodreads",
			"fab-f3a9" => "fab fa-goodreads-g",
			"fab-f1a0" => "fab fa-google",
			"fab-f3aa" => "fab fa-google-drive",
			"fab-f3ab" => "fab fa-google-play",
			"fab-f2b3" => "fab fa-google-plus",
			"fab-f0d5" => "fab fa-google-plus-g",
			"fab-f0d4" => "fab fa-google-plus-square",
			"fab-f1ee" => "fab fa-google-wallet",
			"fab-f184" => "fab fa-gratipay",
			"fab-f2d6" => "fab fa-grav",
			"fab-f3ac" => "fab fa-gripfire",
			"fab-f3ad" => "fab fa-grunt",
			"fab-f3ae" => "fab fa-gulp",
			"fab-f1d4" => "fab fa-hacker-news",
			"fab-f3af" => "fab fa-hacker-news-square",
			"fab-f5f7" => "fab fa-hackerrank",
			"fab-f452" => "fab fa-hips",
			"fab-f3b0" => "fab fa-hire-a-helper",
			"fab-f427" => "fab fa-hooli",
			"fab-f592" => "fab fa-hornbill",
			"fab-f3b1" => "fab fa-hotjar",
			"fab-f27c" => "fab fa-houzz",
			"fab-f13b" => "fab fa-html5",
			"fab-f3b2" => "fab fa-hubspot",
			"fab-f913" => "fab fa-ideal",
			"fab-f2d8" => "fab fa-imdb",
			"fab-f16d" => "fab fa-instagram",
			"fab-f955" => "fab fa-instagram-square",
			"fab-f7af" => "fab fa-intercom",
			"fab-f26b" => "fab fa-internet-explorer",
			"fab-f7b0" => "fab fa-invision",
			"fab-f208" => "fab fa-ioxhost",
			"fab-f83a" => "fab fa-itch-io",
			"fab-f3b4" => "fab fa-itunes",
			"fab-f3b5" => "fab fa-itunes-note",
			"fab-f4e4" => "fab fa-java",
			"fab-f50e" => "fab fa-jedi-order",
			"fab-f3b6" => "fab fa-jenkins",
			"fab-f7b1" => "fab fa-jira",
			"fab-f3b7" => "fab fa-joget",
			"fab-f1aa" => "fab fa-joomla",
			"fab-f3b8" => "fab fa-js",
			"fab-f3b9" => "fab fa-js-square",
			"fab-f1cc" => "fab fa-jsfiddle",
			"fab-f5fa" => "fab fa-kaggle",
			"fab-f4f5" => "fab fa-keybase",
			"fab-f3ba" => "fab fa-keycdn",
			"fab-f3bb" => "fab fa-kickstarter",
			"fab-f3bc" => "fab fa-kickstarter-k",
			"fab-f42f" => "fab fa-korvue",
			"fab-f3bd" => "fab fa-laravel",
			"fab-f202" => "fab fa-lastfm",
			"fab-f203" => "fab fa-lastfm-square",
			"fab-f212" => "fab fa-leanpub",
			"fab-f41d" => "fab fa-less",
			"fab-f3c0" => "fab fa-line",
			"fab-f08c" => "fab fa-linkedin",
			"fab-f0e1" => "fab fa-linkedin-in",
			"fab-f2b8" => "fab fa-linode",
			"fab-f17c" => "fab fa-linux",
			"fab-f3c3" => "fab fa-lyft",
			"fab-f3c4" => "fab fa-magento",
			"fab-f59e" => "fab fa-mailchimp",
			"fab-f50f" => "fab fa-mandalorian",
			"fab-f60f" => "fab fa-markdown",
			"fab-f4f6" => "fab fa-mastodon",
			"fab-f136" => "fab fa-maxcdn",
			"fab-f8ca" => "fab fa-mdb",
			"fab-f3c6" => "fab fa-medapps",
			"fab-f23a" => "fab fa-medium",
			"fab-f3c7" => "fab fa-medium-m",
			"fab-f3c8" => "fab fa-medrt",
			"fab-f2e0" => "fab fa-meetup",
			"fab-f5a3" => "fab fa-megaport",
			"fab-f7b3" => "fab fa-mendeley",
			"fab-f91a" => "fab fa-microblog",
			"fab-f3ca" => "fab fa-microsoft",
			"fab-f3cb" => "fab fa-mix",
			"fab-f289" => "fab fa-mixcloud",
			"fab-f956" => "fab fa-mixer",
			"fab-f3cc" => "fab fa-mizuni",
			"fab-f285" => "fab fa-modx",
			"fab-f3d0" => "fab fa-monero",
			"fab-f3d2" => "fab fa-napster",
			"fab-f612" => "fab fa-neos",
			"fab-f5a8" => "fab fa-nimblr",
			"fab-f419" => "fab fa-node",
			"fab-f3d3" => "fab fa-node-js",
			"fab-f3d4" => "fab fa-npm",
			"fab-f3d5" => "fab fa-ns8",
			"fab-f3d6" => "fab fa-nutritionix",
			"fab-f263" => "fab fa-odnoklassniki",
			"fab-f264" => "fab fa-odnoklassniki-square",
			"fab-f510" => "fab fa-old-republic",
			"fab-f23d" => "fab fa-opencart",
			"fab-f19b" => "fab fa-openid",
			"fab-f26a" => "fab fa-opera",
			"fab-f23c" => "fab fa-optin-monster",
			"fab-f8d2" => "fab fa-orcid",
			"fab-f41a" => "fab fa-osi",
			"fab-f3d7" => "fab fa-page4",
			"fab-f18c" => "fab fa-pagelines",
			"fab-f3d8" => "fab fa-palfed",
			"fab-f3d9" => "fab fa-patreon",
			"fab-f1ed" => "fab fa-paypal",
			"fab-f704" => "fab fa-penny-arcade",
			"fab-f3da" => "fab fa-periscope",
			"fab-f3db" => "fab fa-phabricator",
			"fab-f3dc" => "fab fa-phoenix-framework",
			"fab-f511" => "fab fa-phoenix-squadron",
			"fab-f457" => "fab fa-php",
			"fab-f2ae" => "fab fa-pied-piper",
			"fab-f1a8" => "fab fa-pied-piper-alt",
			"fab-f4e5" => "fab fa-pied-piper-hat",
			"fab-f1a7" => "fab fa-pied-piper-pp",
			"fab-f91e" => "fab fa-pied-piper-square",
			"fab-f0d2" => "fab fa-pinterest",
			"fab-f231" => "fab fa-pinterest-p",
			"fab-f0d3" => "fab fa-pinterest-square",
			"fab-f3df" => "fab fa-playstation",
			"fab-f288" => "fab fa-product-hunt",
			"fab-f3e1" => "fab fa-pushed",
			"fab-f3e2" => "fab fa-python",
			"fab-f1d6" => "fab fa-qq",
			"fab-f459" => "fab fa-quinscape",
			"fab-f2c4" => "fab fa-quora",
			"fab-f4f7" => "fab fa-r-project",
			"fab-f7bb" => "fab fa-raspberry-pi",
			"fab-f2d9" => "fab fa-ravelry",
			"fab-f41b" => "fab fa-react",
			"fab-f75d" => "fab fa-reacteurope",
			"fab-f4d5" => "fab fa-readme",
			"fab-f1d0" => "fab fa-rebel",
			"fab-f3e3" => "fab fa-red-river",
			"fab-f1a1" => "fab fa-reddit",
			"fab-f281" => "fab fa-reddit-alien",
			"fab-f1a2" => "fab fa-reddit-square",
			"fab-f7bc" => "fab fa-redhat",
			"fab-f18b" => "fab fa-renren",
			"fab-f3e6" => "fab fa-replyd",
			"fab-f4f8" => "fab fa-researchgate",
			"fab-f3e7" => "fab fa-resolving",
			"fab-f5b2" => "fab fa-rev",
			"fab-f3e8" => "fab fa-rocketchat",
			"fab-f3e9" => "fab fa-rockrms",
			"fab-f267" => "fab fa-safari",
			"fab-f83b" => "fab fa-salesforce",
			"fab-f41e" => "fab fa-sass",
			"fab-f3ea" => "fab fa-schlix",
			"fab-f28a" => "fab fa-scribd",
			"fab-f3eb" => "fab fa-searchengin",
			"fab-f2da" => "fab fa-sellcast",
			"fab-f213" => "fab fa-sellsy",
			"fab-f3ec" => "fab fa-servicestack",
			"fab-f214" => "fab fa-shirtsinbulk",
			"fab-f957" => "fab fa-shopify",
			"fab-f5b5" => "fab fa-shopware",
			"fab-f215" => "fab fa-simplybuilt",
			"fab-f3ee" => "fab fa-sistrix",
			"fab-f512" => "fab fa-sith",
			"fab-f7c6" => "fab fa-sketch",
			"fab-f216" => "fab fa-skyatlas",
			"fab-f17e" => "fab fa-skype",
			"fab-f198" => "fab fa-slack",
			"fab-f3ef" => "fab fa-slack-hash",
			"fab-f1e7" => "fab fa-slideshare",
			"fab-f2ab" => "fab fa-snapchat",
			"fab-f2ac" => "fab fa-snapchat-ghost",
			"fab-f2ad" => "fab fa-snapchat-square",
			"fab-f1be" => "fab fa-soundcloud",
			"fab-f7d3" => "fab fa-sourcetree",
			"fab-f3f3" => "fab fa-speakap",
			"fab-f83c" => "fab fa-speaker-deck",
			"fab-f1bc" => "fab fa-spotify",
			"fab-f5be" => "fab fa-squarespace",
			"fab-f18d" => "fab fa-stack-exchange",
			"fab-f16c" => "fab fa-stack-overflow",
			"fab-f842" => "fab fa-stackpath",
			"fab-f3f5" => "fab fa-staylinked",
			"fab-f1b6" => "fab fa-steam",
			"fab-f1b7" => "fab fa-steam-square",
			"fab-f3f6" => "fab fa-steam-symbol",
			"fab-f3f7" => "fab fa-sticker-mule",
			"fab-f428" => "fab fa-strava",
			"fab-f429" => "fab fa-stripe",
			"fab-f42a" => "fab fa-stripe-s",
			"fab-f3f8" => "fab fa-studiovinari",
			"fab-f1a4" => "fab fa-stumbleupon",
			"fab-f1a3" => "fab fa-stumbleupon-circle",
			"fab-f2dd" => "fab fa-superpowers",
			"fab-f3f9" => "fab fa-supple",
			"fab-f7d6" => "fab fa-suse",
			"fab-f8e1" => "fab fa-swift",
			"fab-f83d" => "fab fa-symfony",
			"fab-f4f9" => "fab fa-teamspeak",
			"fab-f2c6" => "fab fa-telegram",
			"fab-f3fe" => "fab fa-telegram-plane",
			"fab-f1d5" => "fab fa-tencent-weibo",
			"fab-f69d" => "fab fa-the-red-yeti",
			"fab-f5c6" => "fab fa-themeco",
			"fab-f2b2" => "fab fa-themeisle",
			"fab-f731" => "fab fa-think-peaks",
			"fab-f513" => "fab fa-trade-federation",
			"fab-f181" => "fab fa-trello",
			"fab-f262" => "fab fa-tripadvisor",
			"fab-f173" => "fab fa-tumblr",
			"fab-f174" => "fab fa-tumblr-square",
			"fab-f1e8" => "fab fa-twitch",
			"fab-f099" => "fab fa-twitter",
			"fab-f081" => "fab fa-twitter-square",
			"fab-f42b" => "fab fa-typo3",
			"fab-f402" => "fab fa-uber",
			"fab-f7df" => "fab fa-ubuntu",
			"fab-f403" => "fab fa-uikit",
			"fab-f8e8" => "fab fa-umbraco",
			"fab-f404" => "fab fa-uniregistry",
			"fab-f949" => "fab fa-unity",
			"fab-f405" => "fab fa-untappd",
			"fab-f7e0" => "fab fa-ups",
			"fab-f287" => "fab fa-usb",
			"fab-f7e1" => "fab fa-usps",
			"fab-f407" => "fab fa-ussunnah",
			"fab-f408" => "fab fa-vaadin",
			"fab-f237" => "fab fa-viacoin",
			"fab-f2a9" => "fab fa-viadeo",
			"fab-f2aa" => "fab fa-viadeo-square",
			"fab-f409" => "fab fa-viber",
			"fab-f40a" => "fab fa-vimeo",
			"fab-f194" => "fab fa-vimeo-square",
			"fab-f27d" => "fab fa-vimeo-v",
			"fab-f1ca" => "fab fa-vine",
			"fab-f189" => "fab fa-vk",
			"fab-f40b" => "fab fa-vnv",
			"fab-f41f" => "fab fa-vuejs",
			"fab-f83f" => "fab fa-waze",
			"fab-f5cc" => "fab fa-weebly",
			"fab-f18a" => "fab fa-weibo",
			"fab-f1d7" => "fab fa-weixin",
			"fab-f232" => "fab fa-whatsapp",
			"fab-f40c" => "fab fa-whatsapp-square",
			"fab-f40d" => "fab fa-whmcs",
			"fab-f266" => "fab fa-wikipedia-w",
			"fab-f17a" => "fab fa-windows",
			"fab-f5cf" => "fab fa-wix",
			"fab-f730" => "fab fa-wizards-of-the-coast",
			"fab-f514" => "fab fa-wolf-pack-battalion",
			"fab-f19a" => "fab fa-wordpress",
			"fab-f411" => "fab fa-wordpress-simple",
			"fab-f297" => "fab fa-wpbeginner",
			"fab-f2de" => "fab fa-wpexplorer",
			"fab-f298" => "fab fa-wpforms",
			"fab-f3e4" => "fab fa-wpressr",
			"fab-f412" => "fab fa-xbox",
			"fab-f168" => "fab fa-xing",
			"fab-f169" => "fab fa-xing-square",
			"fab-f23b" => "fab fa-y-combinator",
			"fab-f19e" => "fab fa-yahoo",
			"fab-f840" => "fab fa-yammer",
			"fab-f413" => "fab fa-yandex",
			"fab-f414" => "fab fa-yandex-international",
			"fab-f7e3" => "fab fa-yarn",
			"fab-f1e9" => "fab fa-yelp",
			"fab-f2b1" => "fab fa-yoast",
			"fab-f167" => "fab fa-youtube",
			"fab-f431" => "fab fa-youtube-square",
			"fab-f63f" => "fab fa-zhihu"
		);

		$icons = array_merge( $solid_icons, $regular_icons, $brand_icons );

		$icons = apply_filters( "megamenu_fontawesome_5_icons", $icons );

		return $icons;

	}

	/*public function fa_icons () {
	    $content = file_get_contents('https://raw.githubusercontent.com/FortAwesome/Font-Awesome/master/metadata/icons.json');
	    $json = json_decode($content);
	    $icons = [];

	    foreach ($json as $icon => $value) {
	        foreach ($value->styles as $style) {
	        	$unicode = $value->unicode;
	        	$style = "fa" . substr($style, 0 ,1);
	            echo "'fa-{$unicode}' => '{$style} fa-{$icon}'," . "<br />";
	        }
	    }
	}*/
}

endif;